using System.Collections.Generic;
using System.Linq;
using Modules.DataRetrieval;
using UnityEngine;
using TMPro;

namespace Modules.MachineRecognition.Scripts
{
    /// <summary>
    /// Specifies the type of data to be handled and displayed by the <see cref="PanelManager"/>.
    /// </summary>
    /// <para>
    /// <see cref="SensorData"/> represents real-time sensor readings from machines,
    /// </para>
    /// <para>
    /// <see cref="FuzzyLogicData"/> represents processed data using fuzzy logic for decision-making.
    /// </para>
    public enum DataType
    {
        /// <summary>
        /// Represents real-time sensor readings, such as temperature, pressure, or other machine metrics.
        /// </summary>
        SensorData,

        /// <summary>
        /// Represents processed data using fuzzy logic, typically including machine priority lists or scores for decision-making.
        /// </summary>
        FuzzyLogicData
    }

    /// <summary>
    /// Manages the UI panel for displaying real-time machine data in Unity.
    /// Handles subscription to Server-Sent Events (SSE) for different data types (sensor data or fuzzy logic data),
    /// updates the UI display with incoming data, and ensures proper cleanup of event subscriptions and SSE clients.
    /// </summary>
    public class PanelManager : MonoBehaviour
    {
        /// <summary>
        /// The <see cref="TextMeshPro"/> UI elements used to display data.
        /// </summary>
        [Header("UI Display")]
        [SerializeField] private TextMeshPro machineIdText;
        [SerializeField] private TextMeshPro sensorText1;
        [SerializeField] private TextMeshPro sensorText2;
        [SerializeField] private TextMeshPro sensorText3;
        [SerializeField] private TextMeshPro fuzzyText;
        [SerializeField] private TextMeshPro nameText;
        [SerializeField] private TextMeshPro descText;

        /// <summary>
        /// A reference to the SensorGraphController component.
        /// Used to manage and update the graphical representation of sensor data.
        /// </summary>
        [SerializeField] private SensorGraphController sensorGraph;

        /// <summary>
        /// The type of data to handle and display (<see cref="SensorData"/> or <see cref="FuzzyLogicData"/>).
        /// </summary>
        [Header("Settings")]
        public DataType dataType { get; private set; } = DataType.SensorData;

        /// <summary>
        /// Indicates whether the panel is currently subscribed to SSE events.
        /// </summary>
        private bool _isSubscribed;

        /// <summary>
        /// The SSE endpoint URL to subscribe to for data updates.
        /// </summary>
        private string _url = "https://api.arassistant.nl/mqtt/1/%32";

        /// <summary>
        /// A dictionary mapping sensor types to their corresponding TextMeshPro UI elements.
        /// Used to dynamically update the UI display for sensor data.
        /// </summary>
        private Dictionary<string, TextMeshPro> _sensorTextMap;

        /// <summary>
        /// Unity callback invoked when the PanelManager is destroyed.
        /// Ensures that the panel unsubscribes from Server-Sent Events (SSE) and disposes of any active SSE clients,
        /// preventing memory leaks and dangling event subscriptions when the GameObject is destroyed.
        /// </summary>
        private void OnDestroy()
        {
            StopListeningToSse();
        }

        public void Awake()
        {
            _sensorTextMap = new Dictionary<string, TextMeshPro>();
        }

        /// <summary>
        /// Starts listening to Server-Sent Events (SSE) for the specified data type and URL.
        /// This method ensures that the panel subscribes to real-time updates from the server,
        /// creating and starting the appropriate SSE client and subscribing to the event bus
        /// for UI updates. If already subscribed, the method does nothing.
        /// </summary>
        public void StartListeningToSse()
        {
            if (_isSubscribed) return;

            Debug.Log("Starting to listen to SSE for " + dataType + " at URL: " + _url);

            switch (dataType)
            {
                case DataType.SensorData:
                    SSEManager.Instance.GetOrCreateClient<SensorData>(_url);
                    SSEManager.Instance.StartClient<SensorData>(_url);
                    SSEEventBus.Subscribe<SensorData>(_url, UpdateText);
                    break;

                case DataType.FuzzyLogicData:
                    SSEManager.Instance.GetOrCreateClient<FuzzyLogicData>(_url);
                    SSEManager.Instance.StartClient<FuzzyLogicData>(_url);
                    SSEEventBus.Subscribe<FuzzyLogicData>(_url, UpdateText);
                    break;
            }

            _isSubscribed = true;
        }

        /// <summary>
        /// Stops listening to Server-Sent Events (SSE) for the current data type and URL.
        /// This method unsubscribes the panel from the SSE event bus, disposes of the active SSE client,
        /// and updates the subscription state. If the panel is not currently subscribed, the method does nothing.
        /// </summary>
        public void StopListeningToSse()
        {
            if (!_isSubscribed) return;

            switch (dataType)
            {
                case DataType.SensorData:
                    SSEEventBus.Unsubscribe<SensorData>(_url, UpdateText);
                    break;
                case DataType.FuzzyLogicData:
                    SSEEventBus.Unsubscribe<FuzzyLogicData>(_url, UpdateText);
                    break;
            }

            SSEManager.Instance.DisposeClient(_url);
            _isSubscribed = false;
        }

        /// <summary>
        /// Updates the UI text elements and sensor graph with the provided sensor data.
        /// </summary>
        /// <param name="data">
        /// The sensor data to display, containing the sensor type and its value.
        /// If the data is null, a default message is displayed.
        /// </param>
        private void UpdateText(SensorData data)
        {
            if (data == null)
            {
                sensorText1.text = "No sensor data received.";
                return;
            }

            if (!_sensorTextMap.TryGetValue(data.sensorType, out var textMesh))
            {
                var availableTexts = new[] { sensorText1, sensorText2, sensorText3 };
                textMesh = availableTexts.FirstOrDefault(t => !_sensorTextMap.ContainsValue(t)) ?? sensorText1;

                _sensorTextMap[data.sensorType] = textMesh;
            }

            textMesh.text = $"{data.sensorType}: {data.value}";
            sensorGraph?.UpdateSensor(data.sensorType, data.value);
        }

        /// <summary>
        /// Updates the text display with the received fuzzy logic data.
        /// Formats and displays a list of machines with their priorities and scores,
        /// or shows a message if no machine data is available.
        /// </summary>
        /// <param name="data">
        /// The fuzzy logic data to display. Contains a priority list of machines,
        /// each with a machine ID, priority, and score.
        /// </param>
        private void UpdateText(FuzzyLogicData data)
        {
            if (data?.priorityList == null || data.priorityList.Count == 0)
            {
                fuzzyText.text = "No machine data received.";
                return;
            }

            string displayText = "Machines:\n";
            foreach (var machine in data.priorityList)
            {
                displayText += $"ID: {machine.machineId} | Priority: {machine.priority} | Score: {machine.score:F1}\n";
            }

            fuzzyText.text = displayText;
        }

        /// <summary>
        /// Updates the machine information displayed on the panel.
        /// </summary>
        /// <param name="machineId">The unique identifier of the machine.</param>
        /// <param name="name">The name of the machine.</param>
        /// <param name="description">A brief description of the machine.</param>
        public void UpdateMachineInfo(int machineId, string name, string description)
        {
            machineIdText.text = $"#{machineId}";
            nameText.text = name;
            descText.text = description;
        }

        /// <summary>
        /// Sets the URL for the Server-Sent Events (SSE) client and restarts the subscription.
        /// </summary>
        /// <param name="url">The new SSE endpoint URL.</param>
        public void SetUrl(string url)
        {
            _url = url;

            if (_isSubscribed) {
                StopListeningToSse();
            }
            StartListeningToSse();
        }

        /// <summary>
        /// Sets the type of data to be handled and displayed by the panel.
        /// Restarts the SSE subscription to reflect the new data type.
        /// </summary>
        /// <param name="type">The new data type (<see cref="DataType.SensorData"/> or <see cref="DataType.FuzzyLogicData"/>).</param>
        public void SetDataType(DataType type)
        {
            dataType = type;

            if (_isSubscribed) {
                StopListeningToSse();
            }
            StartListeningToSse();
        }
    }
}